;;;
;;; Copyright (c) 2010, Lorenz Moesenlechner <moesenle@in.tum.de>
;;; All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions are met:
;;; 
;;;     * Redistributions of source code must retain the above copyright
;;;       notice, this list of conditions and the following disclaimer.
;;;     * Redistributions in binary form must reproduce the above copyright
;;;       notice, this list of conditions and the following disclaimer in the
;;;       documentation and/or other materials provided with the distribution.
;;;     * Neither the name of the Intelligent Autonomous Systems Group/
;;;       Technische Universitaet Muenchen nor the names of its contributors 
;;;       may be used to endorse or promote products derived from this software 
;;;       without specific prior written permission.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
;;; AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
;;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
;;; CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
;;; ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
;;; POSSIBILITY OF SUCH DAMAGE.
;;;

(in-package :ct)

(defvar *left-gripper-obj* nil)
(defvar *right-gripper-obj* nil)

(defun attach-obj (side obj)
  "Attaches the the CollisionObject `obj' to the gripper at
`side'. This function is used to store information that is required
later on in the plan, for instance for putting down."
  (ecase side
    (:left (setf *left-gripper-obj* obj))
    (:right (setf *right-gripper-obj* obj))))

(defun get-obj (side)
  (ecase side
    (:left *left-gripper-obj*)
    (:right *right-gripper-obj*)))

(defun remove-obj (side)
  "Removes the object from the gripper"
  (ecase side
    (:left (setf *left-gripper-obj* nil))
    (:right (setf *right-gripper-obj* nil))))

(defun pose-stamped->msg (ps)
  "Converts a tf:pose-stamped instance to the corresponding message
type geometry_msgs/PoseStamped."
  (roslisp:make-message "geometry_msgs/PoseStamped"
                        (stamp header) (tf:stamp ps)
                        (frame_id header) (tf:frame-id ps)
                        (x position pose) (cl-transforms:x
                                           (cl-transforms:origin ps))
                        (y position pose) (cl-transforms:y
                                           (cl-transforms:origin ps))
                        (z position pose) (cl-transforms:z
                                           (cl-transforms:origin ps))
                        (x orientation pose) (cl-transforms:x
                                              (cl-transforms:orientation ps))
                        (y orientation pose) (cl-transforms:y
                                              (cl-transforms:orientation ps))
                        (z orientation pose) (cl-transforms:z
                                              (cl-transforms:orientation ps))
                        (w orientation pose) (cl-transforms:w
                                              (cl-transforms:orientation ps))))

(defun msg->pose-stamped (msg)
  "Converts a geometry_msgs/PoseStamped to a tf:pose-stamped"
  (with-fields
      ((frame-id (frame_id header))
       (stamp (stamp header))
       (x (x position pose))
       (y (y position pose))
       (z (z position pose))
       (qx (x orientation pose))
       (qy (y orientation pose))
       (qz (z orientation pose))
       (qw (w orientation pose)))
      msg
    (tf:make-pose-stamped
     frame-id stamp
     (cl-transforms:make-3d-vector x y z)
     (cl-transforms:make-quaternion qx qy qz qw))))

(define-condition action-failure (plan-error)
  ((state :initarg :state :reader state)
   (result :initarg :result :reader result)))

(defun call-action (action goal &key feedback-cb timeout)
  "Calls `action' with `goal' and throws an error if the action did
  not succeed. This is just a simple wrapper around
  actionlib:call-goal to simplify error handling."
  (multiple-value-bind (result state)
      (actionlib:call-goal action goal :feedback-cb feedback-cb :timeout timeout)
    (unless (eq state :succeeded)
      (fail 'action-failure :state state :result result))
    (values result state)))

(defun fix-integer (i &optional (bits 32))
  "This is a workaround for a bug in roslisp. Negative integers are
not deserialized properly."
  (if (and (not (< i 0)) (> (logand i (ash 1 (1- bits))) 0))
      (- i (expt 2 bits))
      i))

(defun motion-planning-error-code (msg)
  "Returns the (fixed) motion planning error code, as a symbol as
defined in MOTION_PLANNING_MSGS-MSG::<ARMNAVIGATIONERRORCODES>"
  (with-fields (val) msg
    (car
     (rassoc (fix-integer val 32)
             (symbol-codes 'motion_planning_msgs-msg::<armnavigationerrorcodes>)))))
